package com.github.theholywaffle.teamspeak3;

/*
 * #%L
 * TeamSpeak 3 Java API
 * %%
 * Copyright (C) 2014 Bert De Geyter
 * %%
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 * #L%
 */

import com.github.theholywaffle.teamspeak3.TS3Query.FloodRate;
import com.github.theholywaffle.teamspeak3.TS3Query.Protocol;
import com.github.theholywaffle.teamspeak3.api.reconnect.ConnectionHandler;
import com.github.theholywaffle.teamspeak3.api.reconnect.ReconnectStrategy;

/**
 * Class used to configure the behavior of a {@link TS3Query}.
 */
public class TS3Config {

	private boolean frozen = false;

	private String host = null;
	private int queryPort = -1;
	private Protocol protocol = Protocol.RAW;
	private String username = null;
	private String password = null;
	private FloodRate floodRate = FloodRate.DEFAULT;
	private boolean enableCommunicationsLogging = false;
	private int commandTimeout = 4000;
	private ReconnectStrategy reconnectStrategy = ReconnectStrategy.disconnect();
	private ConnectionHandler connectionHandler = null;

	/**
	 * Sets the hostname or IP address of the TeamSpeak3 server to connect to.
	 * <p>
	 * Note that the query port <strong>is not</strong> part of the hostname -
	 * use {@link #setQueryPort(int)} for that purpose.
	 * </p><p>
	 * If the application is running on the same machine as the TS3 server, you can use
	 * {@code null} as the hostname. You can also use any other loopback address,
	 * such as {@code localhost} or {@code 127.0.0.1}.
	 * </p>
	 *
	 * @param host
	 * 		a valid hostname or IP address of a TeamSpeak3 server, or {@code null}
	 *
	 * @return this TS3Config object for chaining
	 */
	public TS3Config setHost(String host) {
		checkFrozen();

		this.host = host;
		return this;
	}

	String getHost() {
		return host;
	}

	/**
	 * Sets the query port to use when connecting to the TeamSpeak3 server.
	 * <p>
	 * Note that the query uses a different port to connect to a server than the regular
	 * TeamSpeak3 clients. Regular clients use "voice ports", the query uses the "query port".
	 * </p><p>
	 * If you don't set the query port by calling this method, the query will use the default
	 * query port:
	 * </p>
	 * <ul>
	 *     <li>{@code 10011} when connecting using {@link Protocol#RAW}</li>
	 *     <li>{@code 10022} when connecting using {@link Protocol#SSH}</li>
	 * </ul>
	 *
	 * @param queryPort
	 * 		the query port to use, must be between {@code 1} and {@code 65535}
	 *
	 * @return this TS3Config object for chaining
	 *
	 * @throws IllegalArgumentException
	 * 		if the port is out of range
	 */
	public TS3Config setQueryPort(int queryPort) {
		checkFrozen();

		if (queryPort <= 0 || queryPort > 65535) {
			throw new IllegalArgumentException("Port out of range: " + queryPort);
		}
		this.queryPort = queryPort;
		return this;
	}

	int getQueryPort() {
		if (queryPort > 0) {
			return queryPort;
		} else {
			// Query port not set by user, use default for chosen protocol
			return protocol == Protocol.SSH ? 10022 : 10011;
		}
	}

	/**
	 * Defines the protocol used to connect to the TeamSpeak3 server.
	 * By default, {@link Protocol#RAW} is used.
	 *
	 * @param protocol
	 * 		the connection protocol to use
	 *
	 * @return this TS3Config object for chaining
	 *
	 * @throws IllegalArgumentException
	 * 		if {@code protocol} is {@code null}
	 * @see Protocol Protocol
	 */
	public TS3Config setProtocol(Protocol protocol) {
		checkFrozen();

		if (protocol == null) throw new IllegalArgumentException("protocol cannot be null!");
		this.protocol = protocol;
		return this;
	}

	Protocol getProtocol() {
		return protocol;
	}

	/**
	 * Authenticates the query with the TeamSpeak3 server using the given login credentials
	 * immediately after connecting.
	 * <p>
	 * Setting the login credentials is mandatory when using the {@link Protocol#SSH} protocol.
	 * </p><p>
	 * A server query login can be generated by heading over to the TeamSpeak3 Client, Tools,
	 * ServerQuery Login. Note that the server query will have the same permissions as the client who
	 * generated the credentials.
	 * </p>
	 *
	 * @param username
	 * 		the username used to authenticate the query
	 * @param password
	 * 		the password corresponding to {@code username}
	 *
	 * @return this TS3Config object for chaining
	 */
	public TS3Config setLoginCredentials(String username, String password) {
		checkFrozen();

		this.username = username;
		this.password = password;
		return this;
	}

	boolean hasLoginCredentials() {
		return username != null && password != null;
	}

	String getUsername() {
		return username;
	}

	String getPassword() {
		return password;
	}

	/**
	 * Sets the delay between sending commands.
	 * <p>
	 * If the query's hostname / IP has not been added to the server's {@code query_ip_whitelist.txt},
	 * you need to use {@link FloodRate#DEFAULT} to prevent the query from being flood-banned.
	 * </p><p>
	 * Calling {@link FloodRate#custom} allows you to use a custom command delay if neither
	 * {@link FloodRate#UNLIMITED} nor {@link FloodRate#DEFAULT} fit your needs.
	 * </p>
	 *
	 * @param rate
	 * 		a {@link FloodRate} object that defines the delay between commands
	 *
	 * @return this TS3Config object for chaining
	 *
	 * @throws IllegalArgumentException
	 * 		if {@code rate} is {@code null}
	 * @see FloodRate FloodRate
	 */
	public TS3Config setFloodRate(FloodRate rate) {
		checkFrozen();

		if (rate == null) throw new IllegalArgumentException("rate cannot be null!");
		this.floodRate = rate;
		return this;
	}

	FloodRate getFloodRate() {
		return floodRate;
	}

	/**
	 * Setting this value to {@code true} will log the communication between the
	 * query client and the TS3 server at the {@code DEBUG} level.
	 * <p>
	 * By default, this is turned off to prevent leaking IPs, tokens, passwords, etc.
	 * into the console and / or log files.
	 * </p>
	 *
	 * @param enable
	 * 		whether to log query commands
	 *
	 * @return this TS3Config object for chaining
	 */
	public TS3Config setEnableCommunicationsLogging(boolean enable) {
		checkFrozen();

		enableCommunicationsLogging = enable;
		return this;
	}

	boolean getEnableCommunicationsLogging() {
		return enableCommunicationsLogging;
	}

	/**
	 * Sets how long the query should wait for any response to a command before disconnecting.
	 * <p>
	 * If the query doesn't receive any data from the TeamSpeak server after
	 * having waited for at least {@code commandTimeout} milliseconds, the connection
	 * is considered to be interrupted, and the query will try to reconnect according to
	 * its {@linkplain TS3Config#setReconnectStrategy(ReconnectStrategy) reconnect strategy}.
	 * </p><p>
	 * By default, this timeout is 4000 milliseconds.
	 * </p>
	 *
	 * @param commandTimeout
	 * 		the minimum amount of time to wait for any response, in milliseconds
	 *
	 * @return this TS3Config object for chaining
	 *
	 * @throws IllegalArgumentException
	 * 		if the timeout value is less than or equal to {@code 0}
	 */
	public TS3Config setCommandTimeout(int commandTimeout) {
		checkFrozen();

		if (commandTimeout <= 0) {
			throw new IllegalArgumentException("Timeout value must be greater than 0");
		}

		this.commandTimeout = commandTimeout;
		return this;
	}

	int getCommandTimeout() {
		return commandTimeout;
	}

	/**
	 * Sets what strategy the query uses to reconnect after having been disconnected.
	 * <p>
	 * The different reconnect strategies let you control whether and after which delay the
	 * query will try to reconnect. By default, {@link ReconnectStrategy#disconnect()} is used,
	 * which doesn't try to reconnect and simply stops the query.
	 * </p><p>
	 * Note that when using a reconnect strategy, you probably also want to set the
	 * {@link ConnectionHandler} using {@link TS3Config#setConnectionHandler(ConnectionHandler)}.
	 *
	 * @param reconnectStrategy
	 * 		the reconnect strategy used when the query loses connection
	 *
	 * @return this TS3Config object for chaining
	 *
	 * @see ReconnectStrategy The reconnect strategies
	 * @see ConnectionHandler The connection handler
	 */
	public TS3Config setReconnectStrategy(ReconnectStrategy reconnectStrategy) {
		checkFrozen();

		if (reconnectStrategy == null) throw new IllegalArgumentException("reconnectStrategy cannot be null!");
		this.reconnectStrategy = reconnectStrategy;
		return this;
	}

	ReconnectStrategy getReconnectStrategy() {
		return reconnectStrategy;
	}

	/**
	 * Sets the {@link ConnectionHandler} that defines the query's behaviour
	 * when connecting or disconnecting.
	 * <p>
	 * The following sample code illustrates how a reconnect strategy and connection handler can be
	 * used to print a message to the console every time the query connects or disconnects:
	 * </p>
	 *
	 * <pre>
	 * config.setReconnectStrategy(ReconnectStrategy.exponentialBackoff());
	 * config.setConnectionHandler(new ConnectionHandler() {
	 * 	&#64;Override
	 * 	public void onConnect(TS3Api api) {
	 * 		System.out.println("Successfully connected!");
	 * 	}
	 *
	 * 	&#64;Override
	 * 	public void onDisconnect(TS3Query query) {
	 * 		System.out.println("The query was disconnected!");
	 * 	}
	 * });
	 * </pre>
	 *
	 * @param connectionHandler
	 * 		the {@link ConnectionHandler} object
	 *
	 * @return this TS3Config object for chaining
	 *
	 * @see TS3Config#setReconnectStrategy(ReconnectStrategy)
	 */
	public TS3Config setConnectionHandler(ConnectionHandler connectionHandler) {
		checkFrozen();

		this.connectionHandler = connectionHandler;
		return this;
	}

	ConnectionHandler getConnectionHandler() {
		return connectionHandler;
	}

	TS3Config freeze() {
		frozen = true;
		return this;
	}

	private void checkFrozen() {
		if (frozen) {
			throw new IllegalStateException("TS3Config cannot be modified after being used to create a TS3Query. " +
					"Please make any changes to TS3Config *before* calling TS3Query's constructor.");
		}
	}
}
