/*
 * Copyright 2015 Austin Keener, Michael Ritter, Florian Spieß, and the JDA contributors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.dv8tion.jda.internal.requests.restaction;

import net.dv8tion.jda.api.JDA;
import net.dv8tion.jda.api.entities.Guild;
import net.dv8tion.jda.api.interactions.IntegrationType;
import net.dv8tion.jda.api.interactions.InteractionContextType;
import net.dv8tion.jda.api.interactions.commands.Command;
import net.dv8tion.jda.api.interactions.commands.DefaultMemberPermissions;
import net.dv8tion.jda.api.interactions.commands.build.CommandData;
import net.dv8tion.jda.api.interactions.commands.build.OptionData;
import net.dv8tion.jda.api.interactions.commands.build.SubcommandData;
import net.dv8tion.jda.api.interactions.commands.build.SubcommandGroupData;
import net.dv8tion.jda.api.requests.Request;
import net.dv8tion.jda.api.requests.Response;
import net.dv8tion.jda.api.requests.Route;
import net.dv8tion.jda.api.requests.restaction.CommandEditAction;
import net.dv8tion.jda.api.utils.data.DataObject;
import net.dv8tion.jda.internal.interactions.CommandDataImpl;
import net.dv8tion.jda.internal.interactions.command.CommandImpl;
import net.dv8tion.jda.internal.requests.RestActionImpl;
import net.dv8tion.jda.internal.utils.Checks;
import okhttp3.RequestBody;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.util.Collection;
import java.util.concurrent.TimeUnit;
import java.util.function.BooleanSupplier;

public class CommandEditActionImpl extends RestActionImpl<Command> implements CommandEditAction
{
    private static final String UNDEFINED = "undefined";
    private static final int NAME_SET                 = 1 << 0;
    private static final int DESCRIPTION_SET          = 1 << 1;
    private static final int OPTIONS_SET              = 1 << 2;
    private static final int PERMISSIONS_SET          = 1 << 3;
    private static final int GUILD_ONLY_SET           = 1 << 4;
    private static final int NSFW_SET                 = 1 << 5;
    private static final int INTERACTION_CONTEXTS_SET = 1 << 6;
    private static final int INTEGRATION_TYPES_SET    = 1 << 7;
    private final Guild guild;

    private int mask;
    private CommandDataImpl data;

    public CommandEditActionImpl(JDA api, String id)
    {
        super(api, Route.Interactions.EDIT_COMMAND.compile(api.getSelfUser().getApplicationId(), id));
        this.guild = null;
        this.reset();
    }

    public CommandEditActionImpl(Guild guild, String id)
    {
        super(guild.getJDA(), Route.Interactions.EDIT_GUILD_COMMAND.compile(guild.getJDA().getSelfUser().getApplicationId(), guild.getId(), id));
        this.guild = guild;
        this.reset();
    }

    @Nonnull
    @Override
    public CommandEditAction setCheck(BooleanSupplier checks)
    {
        return (CommandEditAction) super.setCheck(checks);
    }

    @Nonnull
    @Override
    public CommandEditAction deadline(long timestamp)
    {
        return (CommandEditAction) super.deadline(timestamp);
    }

    @Nonnull
    public CommandEditActionImpl withType(Command.Type type)
    {
        this.mask = 0;
        this.data = CommandDataImpl.of(type, UNDEFINED, UNDEFINED);
        return this;
    }

    @Nonnull
    @Override
    public CommandEditAction apply(@Nonnull CommandData commandData)
    {
        Checks.notNull(commandData, "Command Data");
        this.mask = NAME_SET | DESCRIPTION_SET | OPTIONS_SET | PERMISSIONS_SET | GUILD_ONLY_SET | NSFW_SET | INTERACTION_CONTEXTS_SET | INTEGRATION_TYPES_SET;
        this.data = (CommandDataImpl) commandData;
        return this;
    }

    @Nonnull
    @Override
    public CommandEditAction addCheck(@Nonnull BooleanSupplier checks)
    {
        return (CommandEditAction) super.addCheck(checks);
    }

    @Nonnull
    @Override
    public CommandEditAction timeout(long timeout, @Nonnull TimeUnit unit)
    {
        return (CommandEditAction) super.timeout(timeout, unit);
    }

    @Nonnull
    @Override
    public CommandEditAction setName(@Nullable String name)
    {
        if (name == null)
        {
            mask &= ~NAME_SET;
            return this;
        }
        data.setName(name);
        mask |= NAME_SET;
        return this;
    }

    @Nonnull
    @Override
    public CommandEditAction setGuildOnly(boolean guildOnly)
    {
        data.setGuildOnly(guildOnly);
        mask |= GUILD_ONLY_SET;
        return this;
    }

    @Nonnull
    @Override
    public CommandEditAction setContexts(@Nonnull Collection<InteractionContextType> contexts)
    {
        data.setContexts(contexts);
        mask |= INTERACTION_CONTEXTS_SET;
        return this;
    }

    @Nonnull
    @Override
    public CommandEditAction setIntegrationTypes(@Nonnull Collection<IntegrationType> integrationTypes)
    {
        data.setIntegrationTypes(integrationTypes);
        mask |= INTEGRATION_TYPES_SET;
        return this;
    }

    @Nonnull
    @Override
    public CommandEditAction setNSFW(boolean nsfw)
    {
        data.setNSFW(nsfw);
        mask |= NSFW_SET;
        return this;
    }

    @Nonnull
    @Override
    public CommandEditAction setDefaultPermissions(@Nonnull DefaultMemberPermissions permission)
    {
        data.setDefaultPermissions(permission);
        mask |= PERMISSIONS_SET;
        return this;
    }

    @Nonnull
    @Override
    public CommandEditAction setDescription(@Nullable String description)
    {
        if (description == null)
        {
            mask &= ~DESCRIPTION_SET;
            return this;
        }
        data.setDescription(description);
        mask |= DESCRIPTION_SET;
        return this;
    }

    @Nonnull
    @Override
    public CommandEditAction clearOptions()
    {
        data.removeAllOptions();
        mask |= OPTIONS_SET;
        return this;
    }

    @Nonnull
    @Override
    public CommandEditAction addOptions(@Nonnull OptionData... options)
    {
        data.addOptions(options);
        mask |= OPTIONS_SET;
        return this;
    }

    @Nonnull
    @Override
    public CommandEditAction addSubcommands(@Nonnull SubcommandData... subcommands)
    {
        data.addSubcommands(subcommands);
        mask |= OPTIONS_SET;
        return this;
    }

    @Nonnull
    @Override
    public CommandEditAction addSubcommandGroups(@Nonnull SubcommandGroupData... groups)
    {
        data.addSubcommandGroups(groups);
        mask |= OPTIONS_SET;
        return this;
    }

    private boolean isUnchanged(int flag)
    {
        return (mask & flag) != flag;
    }

    @Override
    protected RequestBody finalizeData()
    {
        DataObject json = data.toData();
        if (isUnchanged(NAME_SET))
            json.remove("name");
        if (isUnchanged(DESCRIPTION_SET))
            json.remove("description");
        if (isUnchanged(OPTIONS_SET))
            json.remove("options");
        if (isUnchanged(PERMISSIONS_SET))
            json.remove("default_member_permissions");
        if (isUnchanged(GUILD_ONLY_SET))
            json.remove("dm_permission");
        if (isUnchanged(NSFW_SET))
            json.remove("nsfw");
        if (isUnchanged(INTERACTION_CONTEXTS_SET))
            json.remove("contexts");
        if (isUnchanged(INTEGRATION_TYPES_SET))
            json.remove("integration_types");
        reset();
        return getRequestBody(json);
    }

    @Override
    protected void handleSuccess(Response response, Request<Command> request)
    {
        DataObject json = response.getObject();
        request.onSuccess(new CommandImpl(api, guild, json));
    }

    private void reset()
    {
        mask = 0;
        data = CommandDataImpl.of(data.getType(), UNDEFINED, UNDEFINED);
    }
}
