/*
 * This file is part of adventure, licensed under the MIT License.
 *
 * Copyright (c) 2017-2025 KyoriPowered
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */
package net.kyori.adventure.audience;

import java.util.ArrayList;
import java.util.Collections;
import java.util.function.Consumer;
import java.util.stream.Collector;
import java.util.stream.Collectors;
import net.kyori.adventure.resource.ResourcePackCallback;
import net.kyori.adventure.text.ComponentLike;
import org.jetbrains.annotations.NotNull;

/**
 * {@link Audience}-related utilities.
 *
 * @since 4.13.0
 */
public final class Audiences {
  static final Collector<? super Audience, ?, ForwardingAudience> COLLECTOR = Collectors.collectingAndThen(
    Collectors.toCollection(ArrayList::new),
    audiences -> Audience.audience(Collections.unmodifiableCollection(audiences))
  );

  private Audiences() {
  }

  /**
   * Creates an action to send a message.
   *
   * @param message the message to send
   * @return an action to send a message
   * @since 4.13.0
   */
  public static @NotNull Consumer<? super Audience> sendingMessage(final @NotNull ComponentLike message) {
    return audience -> audience.sendMessage(message);
  }

  static @NotNull ResourcePackCallback unwrapCallback(final Audience forwarding, final Audience dest, final @NotNull ResourcePackCallback cb) {
    if (cb == ResourcePackCallback.noOp()) return cb;

    return (uuid, status, audience) -> {
      cb.packEventReceived(uuid, status, audience == dest ? forwarding : audience);
    };
  }
}
